//go:build release && dev

// В release && dev билде происходит имитация чтения конфига из config.Data, как это происходит в release билде, хотя
// сам конфиг читается из файла, как в debug билде. Это необходимо для проверки работоспособности release билда в
// dev-среде, без необходимости зашивать конфиг в бинарник.
//
// Так же в этом билде подключено полноценное логирование, как в debug билде.
package internal

import (
	"agent/client/config"
	"agent/commons/app"
	"agent/commons/bytes"
	"agent/commons/debug"
	"agent/commons/vars"
	"flag"
	"os"
)

func AppMain() {
	configFile := flag.String("config", vars.DefaultConfigPath, "path to config.json")
	logFile := flag.String("logfile", "./app.log", "path to app.log file (optional)")
	flag.Parse()

	if len(*logFile) == 0 {
		*logFile = "./app.log"
	}

	initLogger(*logFile)

	if err := rewriteConfig(*configFile); err != nil {
		debug.Panic(err)
	}

	config, err := config.Get()
	if err != nil {
		debug.Panic(err)
	}

	ctx, err := app.New(config)
	if err != nil {
		debug.Panic(err)
	}
	app.Run(ctx)
}

func rewriteConfig(configFile string) error {
	fd, err := os.Open(configFile)
	if err != nil {
		return err
	}
	defer fd.Close()

	stat, err := fd.Stat()
	if err != nil {
		return err
	}

	fileSize := stat.Size()
	data := make([]byte, fileSize)
	_, err = fd.Read(data)
	if err != nil {
		return err
	}

	// Генерируем случайный ключ для шифрования кофига
	config.Key = bytes.Random(16)
	// В начало бинарного конфига записываем его размер (4 байта)
	bytes.Set[int32](config.Data, 0, int32(len(data)), 4)
	// Копируем сам конфиг после его размера
	copy(config.Data[4:], data)
	// Шифруем конфиг
	if data, err = config.EncryptData(config.Data, config.Key); err != nil {
		return err
	}

	return nil
}
