package utime

import "time"

type Duration = time.Duration
type Time = time.Time

type Weekday = time.Weekday
type Month = time.Month

const (
	Nanosecond  = time.Nanosecond
	Microsecond = time.Microsecond
	Millisecond = time.Millisecond
	Second      = time.Second
	Minute      = time.Minute
	Hour        = time.Hour
)

const (
	Monday    = time.Monday
	Tuesday   = time.Tuesday
	Wednesday = time.Wednesday
	Thursday  = time.Thursday
	Friday    = time.Friday
	Saturday  = time.Saturday
	Sunday    = time.Sunday
)

const (
	January   = time.January
	February  = time.February
	March     = time.March
	April     = time.April
	May       = time.May
	June      = time.June
	July      = time.July
	August    = time.August
	September = time.September
	October   = time.October
	November  = time.November
	December  = time.December
)

var correctionTime int64

// SyncTime синхронизирует локальное время с серверным временем srvTime (в наносекундах).
func SyncTime(srvTime int64) {
	if srvTime <= 0 {
		correctionTime = 0
	}
	// Текущее локальное время в наносекундах (UTC)
	localTime := time.Now().UTC().UnixNano()
	// Разница между серверным и локальным временем
	correctionTime = srvTime - localTime
}

// Now всегда возвращает текущее время в UTC
func Now() time.Time {
	if correctionTime == 0 {
		return time.Now().UTC()
	}
	return time.Unix(0, time.Now().UTC().UnixNano()+correctionTime).UTC()
}

// NanoNow всегда возвращает текущее время в наносекундах (UTC)
func NanoNow() int64 {
	return Now().UnixNano()
}

// UnixNow всегда возвращает текущее время в секундах (UTC)
func UnixNow() int {
	return int(Now().Unix())
}

// Unix64Now всегда возвращает текущее время в секундах (UTC)
func Unix64Now() int64 {
	return Now().Unix()
}

// Until возвращает время до указанного времени t
func Until(t time.Time) time.Duration {
	return t.Sub(Now())
}

// Sleep останавливает выполнение текущей горутины на указанное время d
func Sleep(d time.Duration) {
	time.Sleep(d)
}

// Date создает время в UTC с указанными параметрами
func Date(year int, month time.Month, day int, hour, min, sec int) time.Time {
	return time.Date(year, month, day, hour, min, sec, 0, time.UTC)
}
