//go:build debug || dev || sdk

package logger

import (
	"io"
	"log"
	"os"
)

type ignoreWriteErrors struct{ w io.Writer }

func (x ignoreWriteErrors) Write(p []byte) (int, error) {
	_, _ = x.w.Write(p)
	// Возвращаем успех, даже если были ошибки записи
	return len(p), nil
}

type Option func(*config)

type config struct {
	openFlags int
	logFlags  int
}

func WithOpenFlags(flags int) Option {
	return func(c *config) {
		c.openFlags = flags
	}
}

func WithLogFlags(flags int) Option {
	return func(c *config) {
		c.logFlags = flags
	}
}

func New(logFile string, opts ...Option) (*Logger, error) {
	cfg := config{
		openFlags: os.O_CREATE | os.O_WRONLY | os.O_APPEND,
		logFlags:  log.Ldate | log.Ltime | log.LUTC,
	}

	for _, opt := range opts {
		opt(&cfg)
	}

	// Общий файл для логово
	file, err := os.OpenFile(
		logFile,
		cfg.openFlags,
		0644,
	)
	if err != nil {
		return nil, err
	}

	infoWriter := io.MultiWriter(ignoreWriteErrors{os.Stdout}, file)
	errorWriter := io.MultiWriter(ignoreWriteErrors{os.Stderr}, file)

	return &Logger{
		info: log.New(
			infoWriter,
			"[INF] ",
			cfg.logFlags,
		),
		errx: log.New(
			errorWriter,
			"[ERR] ",
			cfg.logFlags,
		),
	}, nil
}
