package api

import (
	"agent/client/events"
	"context"
	"encoding/json"

	"github.com/google/uuid"
)

type UserGroup struct {
	Uuid         uuid.UUID       `json:"uuid"`
	Name         string          `json:"name"`
	ProjectName  string          `json:"project_name"`
	ProjectUuid  uuid.UUID       `json:"project_uuid"`
	Settings     map[string]any  `json:"settings"`
	Admin        bool            `json:"admin"`
	System       bool            `json:"system"`
	Type         int             `json:"type"`
	Rights       map[string]bool `json:"rights"`
	EditorRights map[string]bool `json:"editor_rights"`
	CreatorUuid  uuid.UUID       `json:"creator_uuid"`
	Created      int64           `json:"created"`
	Updated      int64           `json:"updated"`
}

func UserGroupCreate(ctx context.Context, client IClient,
	args UserGroupCreateArgs,
) (res UserGroup, err error) {
	event, err := EventUserGroupCreate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupDelete(ctx context.Context, client IClient,
	uuid uuid.UUID,
) error {
	event, err := EventUserGroupDelete(uuid)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func UserGroupGet(ctx context.Context, client IClient,
	args UserGroupGetArgs,
) (res UserGroup, err error) {
	event, err := EventUserGroupGet(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupGetByUuid(ctx context.Context, client IClient,
	uuid uuid.UUID, raw ...bool,
) (res UserGroup, err error) {
	args := UserGroupGetArgs{
		Uuid: uuid,
	}
	if len(raw) > 0 {
		args.Raw = &raw[0]
	}
	return UserGroupGet(ctx, client, args)
}

func UserGroupGetByName(ctx context.Context, client IClient,
	projectUuid uuid.UUID, name string, raw ...bool,
) (res UserGroup, err error) {
	args := UserGroupGetArgs{
		ProjectUuid: projectUuid,
		Name:        name,
	}
	if len(raw) > 0 {
		args.Raw = &raw[0]
	}
	return UserGroupGet(ctx, client, args)
}

func UserGroupUpdate(ctx context.Context, client IClient,
	args UserGroupUpdateArgs,
) (res UserGroup, err error) {
	event, err := EventUserGroupUpdate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListByProjectAndUser(ctx context.Context, client IClient,
	args UserGroupListByProjectAndUserArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListByProjectAndUser(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListByProject(ctx context.Context, client IClient,
	args UserGroupListByProjectArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListByProject(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListByUser(ctx context.Context, client IClient,
	args UserGroupListByUserArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListByUser(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListByAgentGroups(ctx context.Context, client IClient,
	args UserGroupListByAgentGroupsArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListByAgentGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListByAgent(ctx context.Context, client IClient,
	args UserGroupListByAgentArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListByAgent(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupListAll(ctx context.Context, client IClient,
	args UserGroupListAllArgs,
) (res []UserGroup, err error) {
	event, err := EventUserGroupListAll(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func UserGroupAddUsers(ctx context.Context, client IClient,
	groupUuids []uuid.UUID, userUuids []uuid.UUID,
) error {
	event, err := EventUserGroupAddUsers(groupUuids, userUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func UserGroupDeleteUsers(ctx context.Context, client IClient,
	groupUuids []uuid.UUID, userUuids []uuid.UUID,
) error {
	event, err := EventUserGroupDeleteUsers(groupUuids, userUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}
