package api

import (
	"agent/client/events"
	"context"
	"encoding/json"

	"github.com/google/uuid"
)

type ClientVersion struct {
	ProjectName string `json:"project_name"`
	ProjectUuid UUID   `json:"project_uuid"`
	Uuid        UUID   `json:"uuid"`
	Name        string `json:"name"`
	Value       string `json:"value"`
	CreatorUuid UUID   `json:"creator_uuid"`
	Created     int64  `json:"created"`
	Updated     int64  `json:"updated"`
}

func ClientVersionCreate(ctx context.Context, client IClient,
	args ClientVersionCreateArgs,
) (res ClientVersion, err error) {
	event, err := EventClientVersionCreate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func ClientVersionDelete(ctx context.Context, client IClient,
	uuid uuid.UUID,
) error {
	event, err := EventClientVersionDelete(uuid)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func ClientVersionGet(ctx context.Context, client IClient,
	args ClientVersionGetArgs,
) (res ClientVersion, err error) {
	event, err := EventClientVersionGet(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func ClientVersionGetByUuid(ctx context.Context, client IClient,
	uuid uuid.UUID,
) (res ClientVersion, err error) {
	return ClientVersionGet(ctx, client, ClientVersionGetArgs{
		Uuid: uuid,
	})
}

func ClientVersionGetByName(ctx context.Context, client IClient,
	projectUuid uuid.UUID, name string,
) (res ClientVersion, err error) {
	return ClientVersionGet(ctx, client, ClientVersionGetArgs{
		ProjectUuid: projectUuid,
		Name:        name,
	})
}

func ClientVersionGetByValue(ctx context.Context, client IClient,
	projectUuid uuid.UUID, value string,
) (res ClientVersion, err error) {
	return ClientVersionGet(ctx, client, ClientVersionGetArgs{
		ProjectUuid: projectUuid,
		Value:       value,
	})
}

func ClientVersionUpdate(ctx context.Context, client IClient,
	args ClientVersionUpdateArgs,
) (res ClientVersion, err error) {
	event, err := EventClientVersionUpdate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func ClientVersionList(ctx context.Context, client IClient,
	args ClientVersionListArgs,
) (res []ClientVersion, err error) {
	event, err := EventClientVersionList(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	err = json.Unmarshal(data, &res)
	return res, err
}

func ClientVersionListByProject(ctx context.Context, client IClient,
	args ClientVersionListByProjectArgs,
) (res []ClientVersion, err error) {
	event, err := EventClientVersionListByProject(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func ClientVersionListAll(ctx context.Context, client IClient,
	args ClientVersionListAllArgs,
) (res []ClientVersion, err error) {
	event, err := EventClientVersionListAll(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}
