package api

import (
	"agent/client/events"
	"agent/commons/utils"
	"agent/defines/devent/daction"
	"agent/defines/devent/dcategory"
	"agent/defines/devent/dkind"
)

type ClientVersionCreateArgs struct {
	ProjectUuid UUID   `json:"project_uuid"`
	Uuid        UUID   `json:"uuid,omitempty"`
	Name        string `json:"name"`
	Value       string `json:"value,omitempty"`
}

func EventClientVersionCreate(args ClientVersionCreateArgs) (IEvent, error) {
	meta, err := utils.ConvertTo[map[string]any](args)
	if err != nil {
		return nil, err
	}
	return events.NewEvent(
		dkind.UserToService,
		[]Category{dcategory.Service, dcategory.Client, dcategory.Version},
		daction.Create,
		meta,
	), nil
}

func EventClientVersionDelete(uuid UUID) (IEvent, error) {
	meta := map[string]any{
		"uuid": uuid,
	}
	return events.NewEvent(
		dkind.UserToService,
		[]Category{dcategory.Service, dcategory.Client, dcategory.Version},
		daction.Delete,
		meta,
	), nil
}

type ClientVersionGetArgs struct {
	Uuid        UUID   `json:"uuid,omitempty"`
	ProjectUuid UUID   `json:"project_uuid,omitempty"`
	Value       string `json:"value,omitempty"`
	Name        string `json:"name,omitempty"`
}

func EventClientVersionGet(args ClientVersionGetArgs) (IEvent, error) {
	meta, err := utils.ConvertTo[map[string]any](args)
	if err != nil {
		return nil, err
	}
	return events.NewEvent(
		dkind.UserToService,
		[]Category{dcategory.Service, dcategory.Client, dcategory.Version},
		daction.Get,
		meta,
	), nil
}

type ClientVersionUpdateArgs struct {
	Name  string `json:"name,omitempty"`
	Uuid  UUID   `json:"uuid"`
	Value string `json:"value,omitempty"`
}

func EventClientVersionUpdate(args ClientVersionUpdateArgs) (IEvent, error) {
	meta, err := utils.ConvertTo[map[string]any](args)
	if err != nil {
		return nil, err
	}
	return events.NewEvent(
		dkind.UserToService,
		[]Category{dcategory.Service, dcategory.Client, dcategory.Version},
		daction.Update,
		meta,
	), nil
}

type ClientVersionListArgs struct {
	Kind    string                      `json:"kind"`
	Filter  ClientVersionListFilterOpts `json:"filter,omitempty"`
	Offset  int64                       `json:"offset,omitempty"`
	Limit   int64                       `json:"limit,omitempty"`
	Reverse *bool                       `json:"reverse,omitempty"`
}

type ClientVersionListFilterOpts struct {
	ProjectUuid UUID   `json:"project_uuid,omitempty"`
	Search      string `json:"search,omitempty"`
}

func EventClientVersionList(args ClientVersionListArgs) (IEvent, error) {
	meta, err := utils.ConvertTo[map[string]any](args)
	if err != nil {
		return nil, err
	}
	return events.NewEvent(
		dkind.UserToService,
		[]Category{dcategory.Service, dcategory.Client, dcategory.Version},
		daction.List,
		meta,
	), nil
}

type ClientVersionListByProjectArgs struct {
	ProjectUuid UUID   `json:"project_uuid"`
	Search      string `json:"search,omitempty"`
	Offset      int64  `json:"offset,omitempty"`
	Limit       int64  `json:"limit,omitempty"`
	Reverse     *bool  `json:"reverse,omitempty"`
}

func EventClientVersionListByProject(args ClientVersionListByProjectArgs) (IEvent, error) {
	args_ := ClientVersionListArgs{
		Kind: "by_project",
		Filter: ClientVersionListFilterOpts{
			ProjectUuid: args.ProjectUuid,
			Search:      args.Search,
		},
		Offset: args.Offset,
		Limit:  args.Limit,
	}
	return EventClientVersionList(args_)
}

type ClientVersionListAllArgs struct {
	Search  string `json:"search,omitempty"`
	Offset  int64  `json:"offset,omitempty"`
	Limit   int64  `json:"limit,omitempty"`
	Reverse *bool  `json:"reverse,omitempty"`
}

func EventClientVersionListAll(args ClientVersionListAllArgs) (IEvent, error) {
	args_ := ClientVersionListArgs{
		Kind: "all",
		Filter: ClientVersionListFilterOpts{
			Search: args.Search,
		},
		Offset:  args.Offset,
		Limit:   args.Limit,
		Reverse: args.Reverse,
	}
	return EventClientVersionList(args_)
}
