package api

import (
	"agent/client/events"
	"context"
	"encoding/json"
)

type AgentToken struct {
	Uuid UUID `json:"uuid"`

	AgentUuid     UUID   `json:"agent_uuid"`
	ClientKeyUuid UUID   `json:"client_key_uuid"`
	ClientKeyName string `json:"client_key_name"`
	DeviceUuid    UUID   `json:"device_uuid"`

	TransportFormat int `json:"transport_format"`

	CipherId  int64  `json:"cipher_id"`
	CipherKey []byte `json:"-"`

	Token string `json:"-"`
	Debug bool   `json:"debug"`

	IpAddrList []string `json:"ip_addr_list"`
	IpAddrHash UUID     `json:"ip_addr_hash"`

	Ping  int64 `json:"ping"`
	Count int64 `json:"count"`

	VersionUuid UUID   `json:"version_uuid"`
	Version     string `json:"version"`

	SchemaUuid UUID   `json:"schema_uuid"`
	Schema     string `json:"schema"`

	Status int `json:"status"`

	IntervalCorrection int64 `json:"interval_correction"`

	CommandRound    int64 `json:"command_round"`
	CommandPerRound int64 `json:"command_per_round"`

	PingMinInterval int64 `json:"ping_min_interval"`
	PingMaxInterval int64 `json:"ping_max_interval"`
	PingMinCount    int64 `json:"ping_min_count"`

	NextInterval int64 `json:"next_interval"`

	Expiration     int64 `json:"expiration"`
	ExpirationType int   `json:"expiration_type"`

	Created int64 `json:"created"`
	Updated int64 `json:"updated"`
	Expired int64 `json:"expired"`

	Host      string `json:"host"`
	User      string `json:"user_"`
	Os        string `json:"os"`
	Arch      string `json:"arch"`
	Pid       int64  `json:"pid"`
	Rights    int64  `json:"rights"`
	Image     string `json:"image"`
	Path      string `json:"path"`
	Cwd       string `json:"cwd"`
	SrvSchema string `json:"srv_schema"`
	SrvHost   string `json:"srv_host"`
	SrvPort   int    `json:"srv_port"`
	Name      string `json:"name"`

	DataUpdated int64 `json:"data_updated"`
}

type AgentTokenRaw struct {
	Uuid               UUID     `json:"uuid"`
	AgentUuid          UUID     `json:"agent_uuid"`
	ClientKeyUuid      UUID     `json:"client_key_uuid"`
	DeviceUuid         UUID     `json:"device_uuid"`
	TransportFormat    int      `json:"transport_format"`
	CipherId           int64    `json:"cipher_id"`
	CipherKey          []byte   `json:"-"`
	Token              string   `json:"-"`
	Debug              bool     `json:"debug"`
	IpAddrList         []string `json:"ip_addr_list"`
	IpAddrHash         UUID     `json:"ip_addr_hash"`
	Ping               int64    `json:"ping"`
	Count              int64    `json:"count"`
	VersionUuid        UUID     `json:"version_uuid"`
	SchemaUuid         UUID     `json:"schema_uuid"`
	Status             int      `json:"status"`
	IntervalCorrection int64    `json:"interval_correction"`
	CommandRound       int64    `json:"command_round"`
	CommandPerRound    int64    `json:"command_per_round"`
	PingMinInterval    int64    `json:"ping_min_interval"`
	PingMaxInterval    int64    `json:"ping_max_interval"`
	PingMinCount       int64    `json:"ping_min_count"`
	NextInterval       int64    `json:"next_interval"`
	Expiration         int64    `json:"expiration"`
	ExpirationType     int      `json:"expiration_type"`
	Created            int64    `json:"created"`
	Updated            int64    `json:"updated"`
	Expired            int64    `json:"expired"`

	Version string `json:"version"`
	Schema  string `json:"schema"`
}

type AgentTokenListResult struct {
	NextPage NextPage     `json:"next_page"`
	Items    []AgentToken `json:"items"`
}

type AgentTokenListRawResult struct {
	NextPage NextPage        `json:"next_page"`
	Items    []AgentTokenRaw `json:"items"`
}

func AgentTokenGet(ctx context.Context, client IClient,
	uuid UUID,
) (res AgentToken, err error) {
	event, err := EventAgentTokenGet(uuid)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenGetRaw(ctx context.Context, client IClient,
	uuid UUID,
) (res AgentTokenRaw, err error) {
	event, err := EventAgentTokenGet(uuid, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenDelete(ctx context.Context, client IClient,
	args AgentTokenDeleteArgs,
) error {
	event, err := EventAgentTokenDelete(args)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentTokenListByProjectsAndUserGroups(ctx context.Context, client IClient,
	args AgentTokenListByProjectsAndUserGroupsArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByProjectsAndUserGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByProjectsAndUserGroupsRaw(ctx context.Context, client IClient,
	args AgentTokenListByProjectsAndUserGroupsArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByProjectsAndUserGroups(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByProjectsAndAgentGroups(ctx context.Context, client IClient,
	args AgentTokenListByProjectsAndAgentGroupsArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByProjectsAndAgentGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByProjectsAndAgentGroupsRaw(ctx context.Context, client IClient,
	args AgentTokenListByProjectsAndAgentGroupsArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByProjectsAndAgentGroups(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByProjects(ctx context.Context, client IClient,
	args AgentTokenListByProjectsArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByProjects(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByProjectsRaw(ctx context.Context, client IClient,
	args AgentTokenListByProjectsArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByProjects(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByUserGroups(ctx context.Context, client IClient,
	args AgentTokenListByUserGroupsArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByUserGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByUserGroupsRaw(ctx context.Context, client IClient,
	args AgentTokenListByUserGroupsArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByUserGroups(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByAgentGroups(ctx context.Context, client IClient,
	args AgentTokenListByAgentGroupsArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByAgentGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByAgentGroupsRaw(ctx context.Context, client IClient,
	args AgentTokenListByAgentGroupsArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByAgentGroups(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByAgent(ctx context.Context, client IClient,
	args AgentTokenListByAgentArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListByAgent(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListByAgentRaw(ctx context.Context, client IClient,
	args AgentTokenListByAgentArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListByAgent(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListAll(ctx context.Context, client IClient,
	args AgentTokenListAllArgs,
) (res AgentTokenListResult, err error) {
	event, err := EventAgentTokenListAll(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentTokenListAllRaw(ctx context.Context, client IClient,
	args AgentTokenListAllArgs,
) (res AgentTokenListRawResult, err error) {
	event, err := EventAgentTokenListAll(args, true)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}
