package api

import (
	"agent/client/events"
	"context"
	"encoding/json"
)

type AgentGroup struct {
	Uuid        UUID           `json:"uuid"`
	Name        string         `json:"name"`
	ProjectName string         `json:"project_name"`
	ProjectUuid UUID           `json:"project_uuid"`
	Settings    map[string]any `json:"settings"`
	CreatorUuid UUID           `json:"creator_uuid"`
	Created     int64          `json:"created"`
	Updated     int64          `json:"updated"`
}

func AgentGroupCreate(ctx context.Context, client IClient,
	args AgentGroupCreateArgs,
) (res AgentGroup, err error) {
	event, err := EventAgentGroupCreate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupDelete(ctx context.Context, client IClient,
	uuid UUID,
) error {
	event, err := EventAgentGroupDelete(uuid)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupUpdate(ctx context.Context, client IClient,
	args AgentGroupUpdateArgs,
) (res AgentGroup, err error) {
	event, err := EventAgentGroupUpdate(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupGet(ctx context.Context, client IClient,
	args AgentGroupGetArgs,
) (res AgentGroup, err error) {
	event, err := EventAgentGroupGet(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupGetByUuid(ctx context.Context, client IClient,
	uuid UUID, raw ...bool,
) (res AgentGroup, err error) {
	args := AgentGroupGetArgs{
		Uuid: uuid,
	}
	if len(raw) > 0 {
		args.Raw = &raw[0]
	}
	return AgentGroupGet(ctx, client, args)
}

func AgentGroupGetByName(ctx context.Context, client IClient,
	projectUuid UUID, name string, raw ...bool,
) (res AgentGroup, err error) {
	args := AgentGroupGetArgs{
		ProjectUuid: projectUuid,
		Name:        name,
	}
	if len(raw) > 0 {
		args.Raw = &raw[0]
	}
	return AgentGroupGet(ctx, client, args)
}

func AgentGroupListByProjectAndUserGroups(ctx context.Context, client IClient,
	args AgentGroupListByProjectAndUserGroupsArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByProjectAndUserGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListByProjectAndAgents(ctx context.Context, client IClient,
	args AgentGroupListByProjectAndAgentsArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByProjectAndAgents(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListByProject(ctx context.Context, client IClient,
	args AgentGroupListByProjectArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByProject(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListByUserGroups(ctx context.Context, client IClient,
	args AgentGroupListByUserGroupsArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByUserGroups(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListByClientKey(ctx context.Context, client IClient,
	args AgentGroupListByClientKeyArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByClientKey(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListByAgents(ctx context.Context, client IClient,
	args AgentGroupListByAgentsArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListByAgents(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupListAll(ctx context.Context, client IClient,
	args AgentGroupListAllArgs,
) (res []AgentGroup, err error) {
	event, err := EventAgentGroupListAll(args)
	if err != nil {
		return res, nil
	}
	data, err := events.ExecuteAndGetData(ctx, client, event)
	if err != nil {
		return res, err
	}
	return res, json.Unmarshal(data, &res)
}

func AgentGroupAddAgents(ctx context.Context, client IClient,
	agentGroupUuids []UUID, agentUuids []UUID,
) error {
	event, err := EventAgentGroupAddAgent(agentGroupUuids, agentUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupDeleteAgents(ctx context.Context, client IClient,
	agentGroupUuids []UUID, agentUuids []UUID,
) error {
	event, err := EventAgentGroupDeleteAgent(agentGroupUuids, agentUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupAddUserGroups(ctx context.Context, client IClient,
	agentGroupUuids []UUID, userGroupUuids []UUID,
) error {
	event, err := EventAgentGroupAddUserGroup(agentGroupUuids, userGroupUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupDeleteUserGroups(ctx context.Context, client IClient,
	agentGroupUuids []UUID, userGroupUuids []UUID,
) error {
	event, err := EventAgentGroupDeleteUserGroup(agentGroupUuids, userGroupUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupAddClientKeys(ctx context.Context, client IClient,
	agentGroupUuids []UUID, clientKeyUuids []UUID,
) error {
	event, err := EventAgentGroupAddClientKey(agentGroupUuids, clientKeyUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}

func AgentGroupDeleteClientKeys(ctx context.Context, client IClient,
	agentGroupUuids []UUID, clientKeyUuids []UUID,
) error {
	event, err := EventAgentGroupDeleteClientKey(agentGroupUuids, clientKeyUuids)
	if err != nil {
		return err
	}
	_, err = events.ExecuteAndGetData(ctx, client, event)
	return err
}
